<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Subscription;
use App\Models\Partner;
use Illuminate\Http\Request;
use Carbon\Carbon;

class SubscriptionController extends Controller
{
    public function index(Request $request)
    {
        $query = Subscription::with('partner');

        // Filters
        if ($request->filled('partner_id')) {
            $query->where('partner_id', $request->partner_id);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('plan_type')) {
            $query->where('plan_type', $request->plan_type);
        }

        if ($request->filled('start_date')) {
            $query->where('start_date', '>=', $request->start_date);
        }

        if ($request->filled('end_date')) {
            $query->where('end_date', '<=', $request->end_date);
        }

        $subscriptions = $query->orderBy('created_at', 'desc')->paginate(20);
        $partners = Partner::orderBy('name')->get();

        // Statistics
        $stats = [
            'total_subscriptions' => Subscription::count(),
            'active_subscriptions' => Subscription::where('status', 'active')
                ->where('end_date', '>=', now())
                ->count(),
            'expired_subscriptions' => Subscription::where('status', 'active')
                ->where('end_date', '<', now())
                ->count(),
            'total_revenue' => Subscription::where('status', 'active')->sum('amount'),
            'monthly_revenue' => Subscription::where('status', 'active')
                ->whereMonth('paid_at', now()->month)
                ->whereYear('paid_at', now()->year)
                ->sum('amount'),
        ];

        return view('admin.subscriptions.index', compact('subscriptions', 'partners', 'stats'));
    }

    public function show(Subscription $subscription)
    {
        $subscription->load('partner');
        return view('admin.subscriptions.show', compact('subscription'));
    }

    public function create()
    {
        $partners = Partner::orderBy('name')->get();
        return view('admin.subscriptions.create', compact('partners'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'partner_id' => 'required|exists:partners,id',
            'plan_type' => 'required|in:monthly,yearly',
            'amount' => 'required|numeric|min:0',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'status' => 'required|in:pending,active,expired,cancelled',
        ]);

        // Calculate end date if not provided
        if (!$request->filled('end_date') && $request->filled('start_date')) {
            $startDate = Carbon::parse($request->start_date);
            $validated['end_date'] = $request->plan_type === 'yearly' 
                ? $startDate->copy()->addYear() 
                : $startDate->copy()->addMonth();
        }

        if ($request->status === 'active') {
            $validated['paid_at'] = now();
            
            // Cancel other active subscriptions for this partner
            Subscription::where('partner_id', $validated['partner_id'])
                ->where('status', 'active')
                ->where('id', '!=', $request->id ?? 0)
                ->update(['status' => 'cancelled']);
        }

        Subscription::create($validated);

        return redirect()->route('admin.subscriptions.index')
            ->with('success', 'Subscription created successfully.');
    }

    public function updateStatus(Request $request, Subscription $subscription)
    {
        $validated = $request->validate([
            'status' => 'required|in:pending,active,expired,cancelled',
        ]);

        if ($validated['status'] === 'active') {
            // Cancel other active subscriptions for this partner
            Subscription::where('partner_id', $subscription->partner_id)
                ->where('status', 'active')
                ->where('id', '!=', $subscription->id)
                ->update(['status' => 'cancelled']);

            // Set paid_at if not set
            if (!$subscription->paid_at) {
                $subscription->paid_at = now();
            }
        }

        $subscription->update($validated);

        return redirect()->back()
            ->with('success', 'Subscription status updated successfully.');
    }

    public function revenue(Request $request)
    {
        $query = Subscription::where('status', 'active')->with('partner');

        // Date filters
        if ($request->filled('start_date')) {
            $query->where('paid_at', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('paid_at', '<=', $request->end_date);
        }

        $subscriptions = $query->orderBy('paid_at', 'desc')->get();

        // Monthly revenue breakdown
        $monthlyRevenue = $subscriptions->groupBy(function ($sub) {
            return $sub->paid_at->format('Y-m');
        })->map(function ($group) {
            return [
                'month' => $group->first()->paid_at->format('M Y'),
                'total' => $group->sum('amount'),
                'count' => $group->count(),
            ];
        })->sortKeysDesc();

        // Revenue by plan type
        $revenueByPlan = $subscriptions->groupBy('plan_type')->map(function ($group) {
            return [
                'total' => $group->sum('amount'),
                'count' => $group->count(),
            ];
        });

        // Summary
        $summary = [
            'total_revenue' => $subscriptions->sum('amount'),
            'total_subscriptions' => $subscriptions->count(),
            'monthly_avg' => $monthlyRevenue->count() > 0 
                ? round($subscriptions->sum('amount') / $monthlyRevenue->count(), 2) 
                : 0,
            'this_month' => $subscriptions->where('paid_at', '>=', now()->startOfMonth())->sum('amount'),
        ];

        return view('admin.subscriptions.revenue', compact('subscriptions', 'monthlyRevenue', 'revenueByPlan', 'summary'));
    }

    public function upcomingRenewals(Request $request)
    {
        $days = $request->get('days', 30);
        
        $query = Subscription::where('status', 'active')
            ->where('end_date', '>=', now())
            ->where('end_date', '<=', now()->addDays($days))
            ->with('partner');

        $subscriptions = $query->orderBy('end_date', 'asc')->get();

        // Group by days until renewal
        $renewalsByDays = $subscriptions->groupBy(function ($sub) {
            $daysUntil = now()->diffInDays($sub->end_date, false);
            if ($daysUntil <= 7) return 'within_7_days';
            if ($daysUntil <= 15) return 'within_15_days';
            if ($daysUntil <= 30) return 'within_30_days';
            return 'beyond_30_days';
        });

        $summary = [
            'total_upcoming' => $subscriptions->count(),
            'within_7_days' => $subscriptions->filter(fn($s) => now()->diffInDays($s->end_date, false) <= 7)->count(),
            'within_15_days' => $subscriptions->filter(fn($s) => now()->diffInDays($s->end_date, false) <= 15)->count(),
            'within_30_days' => $subscriptions->filter(fn($s) => now()->diffInDays($s->end_date, false) <= 30)->count(),
            'expected_revenue' => $subscriptions->sum('amount'),
        ];

        return view('admin.subscriptions.upcoming-renewals', compact('subscriptions', 'renewalsByDays', 'summary', 'days'));
    }
}
