<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Loan;
use App\Models\Saving;
use App\Models\Customer;
use App\Models\Repayment;
use App\Models\SavingsCollection;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Carbon\Carbon;

class PartnerApiController extends Controller
{
    /**
     * Login
     */
    public function login(Request $request)
    {
        // Accept both 'username' (for mobile) and 'email' (for web)
        $loginField = $request->has('username') ? 'username' : 'email';
        $loginValue = $request->input($loginField);

        $request->validate([
            $loginField => 'required',
            'password' => 'required',
        ]);

        // Try to find user by email or username (which is also email in our case)
        $user = \App\Models\User::where('email', $loginValue)
            ->whereHas('partner')
            ->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials'
            ], 401);
        }

        $token = $user->createToken('mobile-app')->plainTextToken;

        return response()->json([
            'success' => true,
            'data' => [
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'partner_id' => $user->partner_id,
                    'partner' => $user->partner,
                ],
                'token' => $token,
            ]
        ]);
    }

    /**
     * Logout
     */
    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'success' => true,
            'message' => 'Logged out successfully'
        ]);
    }

    /**
     * Dashboard Summary (Simple stats for mobile)
     */
    public function dashboardSummary(Request $request)
    {
        $partnerId = $request->user()->partner_id;

        $totalLoans = Loan::where('partner_id', $partnerId)->count();
        $activeLoans = Loan::where('partner_id', $partnerId)->where('status', 'active')->count();
        
        $totalSavings = Saving::where('partner_id', $partnerId)->count();
        $activeSavings = Saving::where('partner_id', $partnerId)->where('status', 'active')->count();
        
        $totalCustomers = Customer::where('partner_id', $partnerId)->count();
        
        $totalLoanAmount = Loan::where('partner_id', $partnerId)->sum('loan_amount');
        $totalSavingsAmount = Saving::where('partner_id', $partnerId)->sum('collected_amount');

        return response()->json([
            'success' => true,
            'data' => [
                'total_loans' => $totalLoans,
                'active_loans' => $activeLoans,
                'total_savings' => $totalSavings,
                'active_savings' => $activeSavings,
                'total_customers' => $totalCustomers,
                'total_loan_amount' => (float) $totalLoanAmount,
                'total_savings_amount' => (float) $totalSavingsAmount,
            ]
        ]);
    }

    /**
     * Dashboard Stats (Full dashboard with trends)
     */
    public function dashboard(Request $request)
    {
        $partnerId = $request->user()->partner_id;

        // Total Active Loans
        $activeLoans = Loan::where('partner_id', $partnerId)
            ->where('status', 'active')
            ->sum('loan_amount');

        // Total Savings
        $totalSavings = Saving::where('partner_id', $partnerId)
            ->whereIn('status', ['active', 'matured'])
            ->sum('collected_amount');

        // Repayment Rate
        $totalDue = Repayment::whereHas('loan', function($q) use ($partnerId) {
            $q->where('partner_id', $partnerId);
        })->sum('amount');

        $totalPaid = Repayment::whereHas('loan', function($q) use ($partnerId) {
            $q->where('partner_id', $partnerId);
        })->where('status', 'paid')->sum('amount');

        $repaymentRate = $totalDue > 0 ? ($totalPaid / $totalDue) * 100 : 0;

        // Pending Applications
        $pendingLoans = Loan::where('partner_id', $partnerId)
            ->where('status', 'pending')
            ->count();

        // Recent Activity
        $recentActivity = collect();

        $recentLoans = Loan::where('partner_id', $partnerId)
            ->with('customer')
            ->latest()
            ->take(5)
            ->get()
            ->map(function($loan) {
                return [
                    'type' => 'loan',
                    'id' => $loan->id,
                    'title' => 'New Loan - ' . $loan->customer->name,
                    'amount' => $loan->loan_amount,
                    'status' => $loan->status,
                    'date' => $loan->created_at->format('Y-m-d H:i:s'),
                    'timestamp' => $loan->created_at->timestamp,
                ];
            });

        $recentCollections = SavingsCollection::where('partner_id', $partnerId)
            ->with('customer')
            ->latest()
            ->take(5)
            ->get()
            ->map(function($collection) {
                return [
                    'type' => 'savings_collection',
                    'id' => $collection->id,
                    'title' => 'Savings Deposit - ' . $collection->customer->name,
                    'amount' => $collection->amount,
                    'status' => 'completed',
                    'date' => $collection->created_at->format('Y-m-d H:i:s'),
                    'timestamp' => $collection->created_at->timestamp,
                ];
            });

        $recentRepayments = Repayment::whereHas('loan', function($q) use ($partnerId) {
            $q->where('partner_id', $partnerId);
        })
            ->with('loan.customer')
            ->latest()
            ->take(5)
            ->get()
            ->map(function($repayment) {
                return [
                    'type' => 'repayment',
                    'id' => $repayment->id,
                    'title' => 'Loan Repayment - ' . $repayment->loan->customer->name,
                    'amount' => $repayment->amount,
                    'status' => $repayment->status,
                    'date' => $repayment->created_at->format('Y-m-d H:i:s'),
                    'timestamp' => $repayment->created_at->timestamp,
                ];
            });

        $recentActivity = $recentActivity
            ->merge($recentLoans)
            ->merge($recentCollections)
            ->merge($recentRepayments)
            ->sortByDesc('timestamp')
            ->take(10)
            ->values();

        // 6 Month Trends
        $trends = [];
        for ($i = 5; $i >= 0; $i--) {
            $month = Carbon::now()->subMonths($i);
            $monthStart = $month->copy()->startOfMonth();
            $monthEnd = $month->copy()->endOfMonth();

            $loansAmount = Loan::where('partner_id', $partnerId)
                ->whereBetween('created_at', [$monthStart, $monthEnd])
                ->sum('loan_amount');

            $savingsAmount = SavingsCollection::where('partner_id', $partnerId)
                ->whereBetween('collection_date', [$monthStart, $monthEnd])
                ->sum('amount');

            $trends[] = [
                'month' => $month->format('M'),
                'loans' => (float) $loansAmount,
                'savings' => (float) $savingsAmount,
            ];
        }

        return response()->json([
            'success' => true,
            'data' => [
                'stats' => [
                    'total_active_loans' => (float) $activeLoans,
                    'total_savings' => (float) $totalSavings,
                    'repayment_rate' => round($repaymentRate, 1),
                    'pending_applications' => $pendingLoans,
                ],
                'recent_activity' => $recentActivity,
                'trends' => $trends,
            ]
        ]);
    }

    /**
     * Get Loans
     */
    public function loans(Request $request)
    {
        $partnerId = $request->user()->partner_id;
        $status = $request->query('status'); // active, pending, closed

        $query = Loan::where('partner_id', $partnerId)
            ->with(['customer', 'scheme']);

        if ($status) {
            $query->where('status', $status);
        }

        $loans = $query->latest()->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $loans
        ]);
    }

    /**
     * Get Savings
     */
    public function savings(Request $request)
    {
        $partnerId = $request->user()->partner_id;
        $type = $request->query('type'); // RD, FD, MIS
        $status = $request->query('status'); // active, matured, closed

        $query = Saving::where('partner_id', $partnerId)
            ->with(['customer', 'savingsScheme']);

        if ($type) {
            $query->where('savings_type', $type);
        }

        if ($status) {
            $query->where('status', $status);
        }

        $savings = $query->latest()->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $savings
        ]);
    }

    /**
     * Get Customers
     */
    public function customers(Request $request)
    {
        $partnerId = $request->user()->partner_id;
        $search = $request->query('search');

        $query = Customer::where('partner_id', $partnerId);

        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        $customers = $query->latest()->paginate(50);

        return response()->json([
            'success' => true,
            'data' => $customers
        ]);
    }

    /**
     * Get Schemes
     */
    public function schemes(Request $request)
    {
        $partnerId = $request->user()->partner_id;

        $loanSchemes = \App\Models\Scheme::where('partner_id', $partnerId)
            ->where('status', 'active')
            ->get();

        $savingsSchemes = \App\Models\SavingsScheme::where('partner_id', $partnerId)
            ->where('status', 'active')
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'loan_schemes' => $loanSchemes,
                'savings_schemes' => $savingsSchemes,
            ]
        ]);
    }

    /**
     * User Profile
     */
    public function profile(Request $request)
    {
        $user = $request->user()->load('partner');

        return response()->json([
            'success' => true,
            'data' => $user
        ]);
    }
}

