<?php

namespace App\Http\Controllers\Partner;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\Loan;
use App\Models\Saving;
use App\Models\Repayment;
use App\Models\Penalty;
use Carbon\Carbon;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function index()
    {
        $partnerId = auth()->user()->partner_id;

        // Statistics
        $stats = [
            'total_customers' => Customer::where('partner_id', $partnerId)->count(),
            'total_loans' => Loan::where('partner_id', $partnerId)->count(),
            'active_loans' => Loan::where('partner_id', $partnerId)->where('status', 'active')->count(),
            'total_savings' => Saving::where('partner_id', $partnerId)->count(),
        ];

        // Financial Statistics
        $totalDeposits = Saving::where('partner_id', $partnerId)->where('type', 'deposit')->sum('amount');
        $totalWithdrawals = Saving::where('partner_id', $partnerId)->where('type', 'withdrawal')->sum('amount');
        
        $financials = [
            'total_disbursed' => Loan::where('partner_id', $partnerId)->sum('principal_amount'),
            'total_to_collect' => Loan::where('partner_id', $partnerId)->where('status', 'active')->sum('remaining_amount'),
            'total_collected' => Loan::where('partner_id', $partnerId)->sum('paid_amount'),
            'total_savings_balance' => $totalDeposits - $totalWithdrawals,
        ];

        // Monthly trends (last 6 months)
        $monthlyData = [];
        for ($i = 5; $i >= 0; $i--) {
            $date = Carbon::now()->subMonths($i);
            $monthStart = $date->copy()->startOfMonth();
            $monthEnd = $date->copy()->endOfMonth();
            
            $loansInMonth = Loan::where('partner_id', $partnerId)
                ->whereBetween('disbursement_date', [$monthStart, $monthEnd])
                ->count();
            
            $amountInMonth = Loan::where('partner_id', $partnerId)
                ->whereBetween('disbursement_date', [$monthStart, $monthEnd])
                ->sum('principal_amount');
            
            $monthlyData[] = [
                'month' => $date->format('M'),
                'loans' => $loansInMonth,
                'amount' => $amountInMonth,
            ];
        }

        // Recent loans
        $recentLoans = Loan::where('partner_id', $partnerId)
            ->with(['customer', 'loanScheme', 'scheme'])
            ->latest()
            ->take(5)
            ->get();

        // Status distribution
        $loanStatusData = Loan::where('partner_id', $partnerId)
            ->selectRaw('status, COUNT(*) as count')
            ->groupBy('status')
            ->get();

        // Previous month data for comparison
        $prevMonthStart = Carbon::now()->subMonth()->startOfMonth();
        $prevMonthEnd = Carbon::now()->subMonth()->endOfMonth();
        $currentMonthStart = Carbon::now()->startOfMonth();
        
        $prevMonthLoans = Loan::where('partner_id', $partnerId)
            ->whereBetween('disbursement_date', [$prevMonthStart, $prevMonthEnd])
            ->count();
        
        $currentMonthLoans = Loan::where('partner_id', $partnerId)
            ->whereBetween('disbursement_date', [$currentMonthStart, Carbon::now()])
            ->count();

        $loanGrowth = $prevMonthLoans > 0 
            ? round((($currentMonthLoans - $prevMonthLoans) / $prevMonthLoans) * 100, 1)
            : ($currentMonthLoans > 0 ? 100 : 0);

        $prevMonthAmount = Loan::where('partner_id', $partnerId)
            ->whereBetween('disbursement_date', [$prevMonthStart, $prevMonthEnd])
            ->sum('principal_amount');
        
        $currentMonthAmount = Loan::where('partner_id', $partnerId)
            ->whereBetween('disbursement_date', [$currentMonthStart, Carbon::now()])
            ->sum('principal_amount');

        $amountGrowth = $prevMonthAmount > 0 
            ? round((($currentMonthAmount - $prevMonthAmount) / $prevMonthAmount) * 100, 1)
            : ($currentMonthAmount > 0 ? 100 : 0);

        return view('partner.dashboard', compact(
            'stats',
            'financials',
            'monthlyData',
            'recentLoans',
            'loanStatusData',
            'loanGrowth',
            'amountGrowth'
        ));
    }
}
