<?php

namespace App\Http\Controllers\Partner;

use App\Http\Controllers\Controller;
use App\Models\Penalty;
use App\Models\Loan;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Carbon\Carbon;

class PenaltyController extends Controller
{
    public function index()
    {
        $partnerId = auth()->user()->partner_id;
        $penalties = Penalty::where('partner_id', $partnerId)
            ->with(['loan', 'customer'])
            ->latest()
            ->paginate(15);
        return view('partner.penalties.index', compact('penalties'));
    }

    public function create()
    {
        $partnerId = auth()->user()->partner_id;
        $loans = Loan::where('partner_id', $partnerId)
            ->where('status', 'active')
            ->with('customer')
            ->get();
        return view('partner.penalties.create', compact('loans'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'loan_id' => 'required|exists:loans,id',
            'amount' => 'required|numeric|min:0',
            'penalty_date' => 'required|date',
            'type' => 'required|in:late_payment,default,other',
            'description' => 'nullable|string',
        ]);

        $loan = Loan::findOrFail($validated['loan_id']);
        $this->ensurePartnerOwnership($loan);

        $partnerId = auth()->user()->partner_id;
        $penaltyDate = Carbon::parse($validated['penalty_date']);
        $dueDate = $loan->due_date;
        $daysOverdue = max(0, $penaltyDate->diffInDays($dueDate));

        Penalty::create([
            'partner_id' => $partnerId,
            'loan_id' => $loan->id,
            'customer_id' => $loan->customer_id,
            'penalty_number' => 'PEN-' . strtoupper(Str::random(10)),
            'amount' => $validated['amount'],
            'due_date' => $dueDate,
            'penalty_date' => $penaltyDate,
            'days_overdue' => $daysOverdue,
            'type' => $validated['type'],
            'status' => 'pending',
            'description' => $validated['description'] ?? null,
        ]);

        return redirect()->route('partner.penalties.index')->with('success', 'Penalty created successfully.');
    }

    public function show(Penalty $penalty)
    {
        $this->ensurePartnerOwnership($penalty);
        $penalty->load(['loan', 'customer']);
        return view('partner.penalties.show', compact('penalty'));
    }

    public function update(Request $request, Penalty $penalty)
    {
        $this->ensurePartnerOwnership($penalty);
        
        $validated = $request->validate([
            'status' => 'required|in:pending,paid,waived',
        ]);

        $penalty->update($validated);

        return redirect()->route('partner.penalties.index')->with('success', 'Penalty updated successfully.');
    }

    private function ensurePartnerOwnership($model)
    {
        if ($model->partner_id !== auth()->user()->partner_id) {
            abort(403);
        }
    }
}
