<?php

namespace App\Http\Controllers\Partner;

use App\Http\Controllers\Controller;
use App\Models\Saving;
use App\Models\Customer;
use App\Models\Scheme;
use App\Models\SavingsScheme;
use App\Services\FinancialCalculationService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Carbon\Carbon;

class SavingController extends Controller
{
    public function index()
    {
        $partnerId = auth()->user()->partner_id;
        $savings = Saving::where('partner_id', $partnerId)
            ->with(['customer', 'savingsScheme'])
            ->latest()
            ->paginate(15);
        return view('partner.savings.index', compact('savings'));
    }

    public function create()
    {
        $partnerId = auth()->user()->partner_id;
        $customers = Customer::where('partner_id', $partnerId)->get();
        $savingsSchemes = SavingsScheme::where('partner_id', $partnerId)
            ->where('status', 'active')
            ->get();
        
        return view('partner.savings.create', compact('customers', 'savingsSchemes'));
    }

    public function store(Request $request)
    {
        // Step 1: Validate basic fields
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'savings_scheme_id' => 'required|exists:savings_schemes,id',
            'deposit_date' => 'required|date',
            'notes' => 'nullable|string',
        ]);

        $partnerId = auth()->user()->partner_id;
        $savingsScheme = SavingsScheme::findOrFail($request->savings_scheme_id);
        
        // Ensure scheme belongs to partner
        if ($savingsScheme->partner_id !== $partnerId) {
            return back()->withErrors(['savings_scheme_id' => 'Invalid savings scheme'])->withInput();
        }
        
        // Ensure customer belongs to partner
        $customer = Customer::findOrFail($request->customer_id);
        if ($customer->partner_id !== $partnerId) {
            return back()->withErrors(['customer_id' => 'Invalid customer'])->withInput();
        }

        $savingsType = $savingsScheme->savings_type;
        $depositDate = Carbon::parse($request->deposit_date);
        $durationValue = $savingsScheme->duration_value;
        
        // Step 2: Type-specific validation and processing
        try {
            switch ($savingsType) {
                case 'RD':
                    return $this->createRDAccount($request, $savingsScheme, $customer, $depositDate, $durationValue);
                
                case 'FD':
                    return $this->createFDAccount($request, $savingsScheme, $customer, $depositDate, $durationValue);
                
                case 'MIS':
                    return $this->createMISAccount($request, $savingsScheme, $customer, $depositDate, $durationValue);
                
                default:
                    return back()->withErrors(['savings_scheme_id' => 'Invalid savings type'])->withInput();
            }
        } catch (\Exception $e) {
            \Log::error('Savings account creation failed: ' . $e->getMessage());
            return back()->withErrors(['error' => 'Failed to create savings account: ' . $e->getMessage()])->withInput();
        }
    }

    /**
     * Create RD (Recurring Deposit) Account
     */
    private function createRDAccount(Request $request, SavingsScheme $scheme, Customer $customer, Carbon $depositDate, int $durationValue)
    {
        // Validate RD-specific fields
        $validated = $request->validate([
            'monthly_deposit' => 'required|numeric|min:1',
        ]);
        
        $monthlyDeposit = $validated['monthly_deposit'];
        
        // Validate against scheme limits
        if (!$scheme->validateAmount($monthlyDeposit)) {
            $errorMsg = 'Monthly deposit must be';
            if ($scheme->min_amount) {
                $errorMsg .= ' at least ₹' . number_format($scheme->min_amount, 2);
            }
            if ($scheme->max_amount) {
                $errorMsg .= ' and not more than ₹' . number_format($scheme->max_amount, 2);
            }
            return back()->withErrors(['monthly_deposit' => $errorMsg])->withInput();
        }
        
        // Calculate maturity details
        $durationType = $scheme->duration_type;
        $targetAmount = $monthlyDeposit * $durationValue;
        
        // Calculate maturity date based on duration type
        $maturityDate = FinancialCalculationService::calculateMaturityDate($depositDate, $durationValue, $durationType);
        
        // Calculate RD maturity with proper parameters
        $calculation = FinancialCalculationService::calculateRD(
            $monthlyDeposit,
            $scheme->interest_rate,
            $durationValue,
            $scheme->compounding_frequency ?? 'monthly',
            $scheme->interest_type ?? 'compound',
            $durationType
        );
        
        // Create RD account
        $partner = auth()->user()->partner;
        $saving = Saving::create([
            'partner_id' => $partner->id,
            'customer_id' => $customer->id,
            'savings_scheme_id' => $scheme->id,
            'savings_number' => $partner->generateSavingsNumber(),
            'savings_type' => 'RD',
            'monthly_deposit' => $monthlyDeposit,
            'installment_amount' => $monthlyDeposit,
            'target_amount' => $targetAmount,
            'collected_amount' => 0,
            'interest_rate' => $scheme->interest_rate,
            'interest_type' => $scheme->interest_type,
            'compounding_frequency' => $scheme->compounding_frequency,
            'deposit_date' => $depositDate,
            'duration_type' => $durationType,
            'duration_value' => $durationValue,
            'maturity_date' => $maturityDate,
            'maturity_amount' => $calculation['maturity_amount'],
            'amount' => $targetAmount,
            'type' => 'deposit',
            'status' => 'active',
            'notes' => $request->notes,
        ]);
        
        $frequencyLabel = match($durationType) {
            'daily' => 'daily',
            'weekly' => 'weekly',
            'monthly' => 'monthly',
        };
        
        return redirect()->route('partner.savings.index')->with('success', 
            'RD Account created successfully! ' .
            'Deposit: ₹' . number_format($monthlyDeposit, 2) . ' (' . $frequencyLabel . '), ' .
            'Maturity Amount: ₹' . number_format($calculation['maturity_amount'], 2) . '. ' .
            'Start collecting ' . $frequencyLabel . ' installments from Savings Collections.'
        );
    }

    /**
     * Create FD (Fixed Deposit) Account
     */
    private function createFDAccount(Request $request, SavingsScheme $scheme, Customer $customer, Carbon $depositDate, int $durationValue)
    {
        // Validate FD-specific fields
        $validated = $request->validate([
            'principal_amount' => 'required|numeric|min:1',
            'payment_method' => 'required|in:cash,bank_transfer,cheque,other',
            'reference_number' => 'nullable|string',
        ]);
        
        $principalAmount = $validated['principal_amount'];
        
        // Validate against scheme limits
        if (!$scheme->validateAmount($principalAmount)) {
            $errorMsg = 'Principal amount must be';
            if ($scheme->min_amount) {
                $errorMsg .= ' at least ₹' . number_format($scheme->min_amount, 2);
            }
            if ($scheme->max_amount) {
                $errorMsg .= ' and not more than ₹' . number_format($scheme->max_amount, 2);
            }
            return back()->withErrors(['principal_amount' => $errorMsg])->withInput();
        }
        
        // Calculate maturity details
        $maturityDate = FinancialCalculationService::calculateMaturityDate($depositDate, $durationValue, 'monthly');
        $calculation = FinancialCalculationService::calculateFD(
            $principalAmount,
            $scheme->interest_rate,
            $durationValue,
            $scheme->compounding_frequency ?? 'monthly',
            $scheme->interest_type ?? 'compound'
        );
        
        // Create FD account
        $partner = auth()->user()->partner;
        $saving = Saving::create([
            'partner_id' => $partner->id,
            'customer_id' => $customer->id,
            'savings_scheme_id' => $scheme->id,
            'savings_number' => $partner->generateSavingsNumber(),
            'savings_type' => 'FD',
            'principal_amount' => $principalAmount,
            'target_amount' => $principalAmount,
            'collected_amount' => $principalAmount, // Paid upfront
            'interest_rate' => $scheme->interest_rate,
            'interest_type' => $scheme->interest_type,
            'compounding_frequency' => $scheme->compounding_frequency,
            'deposit_date' => $depositDate,
            'duration_type' => 'monthly',
            'duration_value' => $durationValue,
            'maturity_date' => $maturityDate,
            'maturity_amount' => $calculation['maturity_amount'],
            'amount' => $principalAmount,
            'type' => 'deposit',
            'status' => 'active',
            'notes' => $request->notes,
        ]);
        
        return redirect()->route('partner.savings.index')->with('success', 
            'FD Account created successfully! ' .
            'Principal: ₹' . number_format($principalAmount, 2) . ', ' .
            'Maturity Amount: ₹' . number_format($calculation['maturity_amount'], 2) . ', ' .
            'Interest: ₹' . number_format($calculation['total_interest'], 2) . '. ' .
            'Maturity Date: ' . $maturityDate->format('d M Y')
        );
    }

    /**
     * Create MIS (Monthly Income Scheme) Account
     */
    private function createMISAccount(Request $request, SavingsScheme $scheme, Customer $customer, Carbon $depositDate, int $durationValue)
    {
        // Validate MIS-specific fields
        $validated = $request->validate([
            'principal_amount' => 'required|numeric|min:1',
            'payment_method' => 'required|in:cash,bank_transfer,cheque,other',
            'reference_number' => 'nullable|string',
        ]);
        
        $principalAmount = $validated['principal_amount'];
        
        // Validate against scheme limits
        if (!$scheme->validateAmount($principalAmount)) {
            $errorMsg = 'Principal amount must be';
            if ($scheme->min_amount) {
                $errorMsg .= ' at least ₹' . number_format($scheme->min_amount, 2);
            }
            if ($scheme->max_amount) {
                $errorMsg .= ' and not more than ₹' . number_format($scheme->max_amount, 2);
            }
            return back()->withErrors(['principal_amount' => $errorMsg])->withInput();
        }
        
        // Calculate maturity details
        $maturityDate = FinancialCalculationService::calculateMaturityDate($depositDate, $durationValue, 'monthly');
        $calculation = FinancialCalculationService::calculateMIS(
            $principalAmount,
            $scheme->interest_rate,
            $durationValue
        );
        
        // Create MIS account
        $partner = auth()->user()->partner;
        $saving = Saving::create([
            'partner_id' => $partner->id,
            'customer_id' => $customer->id,
            'savings_scheme_id' => $scheme->id,
            'savings_number' => $partner->generateSavingsNumber(),
            'savings_type' => 'MIS',
            'principal_amount' => $principalAmount,
            'target_amount' => $principalAmount,
            'collected_amount' => $principalAmount, // Paid upfront
            'monthly_interest_payout' => $calculation['monthly_interest_payout'],
            'interest_rate' => $scheme->interest_rate,
            'interest_type' => 'simple',
            'deposit_date' => $depositDate,
            'duration_type' => 'monthly',
            'duration_value' => $durationValue,
            'maturity_date' => $maturityDate,
            'maturity_amount' => $principalAmount, // Principal returned
            'amount' => $principalAmount,
            'type' => 'deposit',
            'status' => 'active',
            'notes' => $request->notes,
        ]);
        
        return redirect()->route('partner.savings.index')->with('success', 
            'MIS Account created successfully! ' .
            'Principal: ₹' . number_format($principalAmount, 2) . ', ' .
            'Monthly Interest Payout: ₹' . number_format($calculation['monthly_interest_payout'], 2) . ', ' .
            'Total Interest: ₹' . number_format($calculation['total_interest_payout'], 2) . '. ' .
            'Maturity Date: ' . $maturityDate->format('d M Y')
        );
    }

    public function show(Saving $saving)
    {
        $this->ensurePartnerOwnership($saving);
        $saving->load(['customer', 'savingsScheme']);
        return view('partner.savings.show', compact('saving'));
    }

    public function statement(Customer $customer)
    {
        $partnerId = auth()->user()->partner_id;
        
        // Ensure customer belongs to partner
        if ($customer->partner_id !== $partnerId) {
            abort(403);
        }

        $savings = Saving::where('customer_id', $customer->id)
            ->where('partner_id', $partnerId)
            ->with(['savingsScheme'])
            ->latest()
            ->get();

        return view('partner.savings.statement', compact('customer', 'savings'));
    }

    private function ensurePartnerOwnership(Saving $saving)
    {
        $partnerId = auth()->user()->partner_id;
        if ($saving->partner_id !== $partnerId) {
            abort(403);
        }
    }
}
