<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Services\FinancialCalculationService;

class Saving extends Model
{
    protected $fillable = [
        'partner_id',
        'customer_id',
        'scheme_id',
        'savings_scheme_id',
        'savings_number',
        'amount',
        'target_amount',
        'collected_amount',
        'installment_amount',
        'principal_amount',
        'monthly_deposit',
        'interest_rate',
        'interest_type',
        'compounding_frequency',
        'duration_type',
        'duration_value',
        'deposit_date',
        'maturity_date',
        'maturity_amount',
        'monthly_interest_payout',
        'savings_type',
        'type',
        'status',
        'notes',
    ];

    protected function casts(): array
    {
        return [
            'amount' => 'decimal:2',
            'target_amount' => 'decimal:2',
            'collected_amount' => 'decimal:2',
            'installment_amount' => 'decimal:2',
            'principal_amount' => 'decimal:2',
            'monthly_deposit' => 'decimal:2',
            'interest_rate' => 'decimal:2',
            'maturity_amount' => 'decimal:2',
            'monthly_interest_payout' => 'decimal:2',
            'deposit_date' => 'date',
            'maturity_date' => 'date',
        ];
    }

    public function partner(): BelongsTo
    {
        return $this->belongsTo(Partner::class);
    }

    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    public function scheme(): BelongsTo
    {
        return $this->belongsTo(Scheme::class);
    }

    public function savingsScheme(): BelongsTo
    {
        return $this->belongsTo(SavingsScheme::class, 'savings_scheme_id');
    }

    public function collections(): HasMany
    {
        return $this->hasMany(SavingsCollection::class);
    }

    public function getDurationDisplayAttribute(): string
    {
        return $this->duration_value . ' ' . ucfirst($this->duration_type);
    }

    public function getRemainingAmountAttribute(): float
    {
        if ($this->savings_type === 'RD' && $this->monthly_deposit) {
            // For RD, calculate remaining based on installments
            $collectedMonths = $this->collections()->count();
            $remainingMonths = max(0, $this->duration_value - $collectedMonths);
            return round($remainingMonths * $this->monthly_deposit, 2);
        } elseif ($this->target_amount) {
            return max(0, $this->target_amount - $this->collected_amount);
        }
        return 0;
    }

    /**
     * Calculate maturity amount based on savings type
     */
    public function calculateMaturityAmount(): float
    {
        if (!$this->savings_type || !$this->interest_rate || !$this->duration_value) {
            return 0;
        }

        $durationMonths = $this->duration_type === 'monthly' ? $this->duration_value : 
                         ($this->duration_type === 'weekly' ? ($this->duration_value * 4) : 
                         ($this->duration_value * 30));

        return match($this->savings_type) {
            'RD' => FinancialCalculationService::calculateRD(
                $this->monthly_deposit ?? 0,
                $this->interest_rate,
                $durationMonths,
                $this->compounding_frequency ?? 'monthly'
            )['maturity_amount'],
            'FD' => FinancialCalculationService::calculateFD(
                $this->principal_amount ?? 0,
                $this->interest_rate,
                $durationMonths,
                $this->compounding_frequency ?? 'monthly'
            )['maturity_amount'],
            'MIS' => $this->principal_amount ?? 0, // Principal returned at maturity
            default => 0
        };
    }

    /**
     * Get monthly interest payout for MIS
     */
    public function getMonthlyInterestPayout(): float
    {
        if ($this->savings_type === 'MIS' && $this->principal_amount && $this->interest_rate) {
            return FinancialCalculationService::calculateMIS(
                $this->principal_amount,
                $this->interest_rate,
                $this->duration_value
            )['monthly_interest_payout'];
        }
        return 0;
    }

    /**
     * Get collected months count (for RD)
     */
    public function getCollectedMonthsAttribute(): int
    {
        return $this->collections()->count();
    }

    /**
     * Get remaining months (for RD)
     */
    public function getRemainingMonthsAttribute(): int
    {
        if ($this->savings_type === 'RD') {
            return max(0, $this->duration_value - $this->collected_months);
        }
        return 0;
    }
}
