<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class SavingsScheme extends Model
{
    protected $fillable = [
        'partner_id',
        'savings_type',
        'name',
        'description',
        'interest_rate',
        'interest_type',
        'compounding_frequency',
        'duration_type',
        'duration_value',
        'min_amount',
        'max_amount',
        'installment_amount',
        'status',
    ];

    protected function casts(): array
    {
        return [
            'interest_rate' => 'decimal:2',
            'min_amount' => 'decimal:2',
            'max_amount' => 'decimal:2',
            'installment_amount' => 'decimal:2',
        ];
    }
    
    /**
     * Validation rules based on savings type
     */
    public static function getValidationRules(string $savingsType): array
    {
        $baseRules = [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'savings_type' => 'required|in:RD,FD,MIS',
            'interest_rate' => 'required|numeric|min:0|max:100',
            'interest_type' => 'required|in:simple,compound',
            'duration_value' => 'required|integer|min:1',
            'status' => 'required|in:active,inactive',
        ];
        
        switch ($savingsType) {
            case 'RD':
                return array_merge($baseRules, [
                    'compounding_frequency' => 'nullable|in:monthly,quarterly,yearly',
                    'installment_amount' => 'required|numeric|min:1',
                    'min_amount' => 'nullable|numeric|min:0',
                    'max_amount' => 'nullable|numeric|gte:min_amount',
                    'duration_type' => 'required|in:daily,weekly,monthly',
                ]);
            
            case 'FD':
                return array_merge($baseRules, [
                    'compounding_frequency' => 'nullable|in:monthly,quarterly,yearly',
                    'min_amount' => 'required|numeric|min:1',
                    'max_amount' => 'nullable|numeric|gte:min_amount',
                    'installment_amount' => 'nullable',
                    'duration_type' => 'required|in:monthly',
                ]);
            
            case 'MIS':
                return array_merge($baseRules, [
                    'compounding_frequency' => 'nullable',
                    'min_amount' => 'required|numeric|min:1',
                    'max_amount' => 'nullable|numeric|gte:min_amount',
                    'installment_amount' => 'nullable',
                    'duration_type' => 'required|in:monthly',
                    'interest_type' => 'required|in:simple',
                ]);
            
            default:
                return $baseRules;
        }
    }
    
    /**
     * Check if scheme is for RD type
     */
    public function isRD(): bool
    {
        return $this->savings_type === 'RD';
    }
    
    /**
     * Check if scheme is for FD type
     */
    public function isFD(): bool
    {
        return $this->savings_type === 'FD';
    }
    
    /**
     * Check if scheme is for MIS type
     */
    public function isMIS(): bool
    {
        return $this->savings_type === 'MIS';
    }
    
    /**
     * Validate amount against scheme limits
     */
    public function validateAmount(float $amount): bool
    {
        if ($this->min_amount && $amount < $this->min_amount) {
            return false;
        }
        
        if ($this->max_amount && $amount > $this->max_amount) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Get frequency label for display
     */
    public function getFrequencyLabel(): string
    {
        if ($this->savings_type === 'RD') {
            return match($this->duration_type) {
                'daily' => 'Daily',
                'weekly' => 'Weekly',
                'monthly' => 'Monthly',
                default => 'Monthly'
            };
        }
        return 'Monthly'; // FD and MIS are always monthly duration
    }

    public function partner(): BelongsTo
    {
        return $this->belongsTo(Partner::class);
    }

    public function savings(): HasMany
    {
        return $this->hasMany(Saving::class, 'savings_scheme_id');
    }
}
