<?php

namespace App\Services;

use App\Models\Loan;
use App\Models\Penalty;
use Carbon\Carbon;
use Illuminate\Support\Str;

class LateFeeService
{
    /**
     * Apply late fees for overdue loans
     */
    public function applyLateFeesForLoan(Loan $loan): void
    {
        if ($loan->status !== 'active') {
            return;
        }

        $scheme = $loan->scheme;
        
        // Check if scheme has late fee configured
        if (!$scheme->late_fee_type || !$scheme->late_fee_value || $scheme->late_fee_value <= 0) {
            return;
        }

        $today = Carbon::today();
        $dueDate = Carbon::parse($loan->due_date);
        
        // Check if payment is overdue
        if ($today->lte($dueDate)) {
            return; // Not overdue yet
        }

        // Check if payment was made after due date
        $paymentAfterDue = $loan->repayments()
            ->where('payment_date', '>', $dueDate)
            ->exists();

        if ($paymentAfterDue) {
            // Payment was made, check if there are still overdue amounts
            // This would require more complex logic based on installment schedule
            // For now, we'll skip if payment was made
            return;
        }

        $daysOverdue = $today->diffInDays($dueDate);
        
        // Check if penalty already exists for this due date and update it if needed
        $existingPenalty = Penalty::where('loan_id', $loan->id)
            ->where('due_date', $dueDate->format('Y-m-d'))
            ->where('status', 'pending')
            ->first();

        // Calculate late fee amount
        $lateFeeAmount = $this->calculateLateFee(
            $scheme->late_fee_type,
            $scheme->late_fee_value,
            $loan->monthly_installment,
            $daysOverdue
        );

        if ($lateFeeAmount <= 0) {
            return;
        }

        if ($existingPenalty) {
            // Update existing penalty with new amount (recalculate based on current days overdue)
            $existingPenalty->update([
                'amount' => round($lateFeeAmount, 2),
                'days_overdue' => $daysOverdue,
                'penalty_date' => $today,
                'description' => "Late fee for overdue payment. {$daysOverdue} day(s) overdue.",
            ]);
        } else {
            // Create new penalty record
            Penalty::create([
                'partner_id' => $loan->partner_id,
                'loan_id' => $loan->id,
                'customer_id' => $loan->customer_id,
                'penalty_number' => 'PEN-' . strtoupper(Str::random(10)),
                'amount' => round($lateFeeAmount, 2),
                'due_date' => $dueDate,
                'penalty_date' => $today,
                'days_overdue' => $daysOverdue,
                'type' => 'late_payment',
                'status' => 'pending',
                'description' => "Late fee for overdue payment. {$daysOverdue} day(s) overdue.",
            ]);
        }
    }

    /**
     * Calculate late fee amount
     */
    private function calculateLateFee(string $type, float $value, float $installmentAmount, int $daysOverdue): float
    {
        return match($type) {
            'percentage' => ($installmentAmount * $value * $daysOverdue) / 100,
            'fixed' => $value * $daysOverdue, // Fixed amount per day
            default => 0,
        };
    }

    /**
     * Apply late fees for all overdue loans
     */
    public function applyLateFeesForAllOverdueLoans(): int
    {
        $count = 0;
        
        $overdueLoans = Loan::where('status', 'active')
            ->where('due_date', '<', Carbon::today())
            ->where('remaining_amount', '>', 0)
            ->with('scheme')
            ->get();

        foreach ($overdueLoans as $loan) {
            try {
                $this->applyLateFeesForLoan($loan);
                $count++;
            } catch (\Exception $e) {
                // Log error but continue with other loans
                \Log::error("Failed to apply late fee for loan {$loan->id}: " . $e->getMessage());
            }
        }

        return $count;
    }
}

